//
//  TodoItemDetail.swift
//  Do It
//
//  Created by Jim Dovey on 9/9/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI

fileprivate extension TodoItem {
    static var empty = TodoItem(title: "", priority: .normal, listID: UUID())
}

struct TodoItemDetail: View {
    @State var item: TodoItem
    @EnvironmentObject var data: DataCenter

    @State var editingItem: TodoItem = .empty
    @State var showingEditor = false
    
    var gradient: LinearGradient {
        LinearGradient(
            gradient: Gradient(colors: [
                Color.black.opacity(0.6),
                Color.black.opacity(0),
            ]),
            startPoint: .bottom,
            endPoint: .init(x: 0.5, y: 0.1)
        )
    }
    
    var headerBackground: some View {
        Rectangle()
            .fill(data.list(for: item).color.uiColor)
            .overlay(gradient)
            .edgesIgnoringSafeArea(.top)
    }

    var cancelButton: some View {
        Button(action: {
            // Close the editor.
            self.showingEditor.toggle()
        }) {
            Text("Cancel")
                .foregroundColor(.accentColor)
        }
        .niceHoverEffect()
    }

    var doneButton: some View {
        Button(action: {
            self.data.updateTodoItem(self.editingItem)
            self.item = self.editingItem
            self.showingEditor.toggle()
        }) {
            Text("Done")
                .bold()
                .foregroundColor(.accentColor)
        }
        .niceHoverEffect()
    }

    var editor: some View {
        NavigationView {
            TodoItemEditor(item: $editingItem)
                .environmentObject(self.data)
                .navigationBarItems(leading: cancelButton,
                                    trailing: doneButton)
        }
    }

    // START:Popover
    var editorButton: some View {
        Button(action: {
            // <literal:elide>show editor</literal:elide>
            // END:Popover
            self.editingItem = self.item
            self.showingEditor.toggle()
            //START:Popover
        }) {
            // <literal:elide>image</literal:elide>
            // END:Popover
            Image(systemName: "square.and.pencil")
                .imageScale(.large)
                .foregroundColor(.accentColor)
            // START:Popover
        }
        .accessibility(label: Text("Edit"))
        .popover(isPresented: $showingEditor, content: {
            self.editor
                .frame(idealWidth: 500, idealHeight: 600)
        })
        // END:Popover
        .iconHoverEffect()
        // START:Popover
    }
    // END:Popover

    var body: some View {
        ScrollView(.vertical) {
            VStack(alignment: .leading) {
                VStack(alignment: .leading, spacing: 8) {
                    Text(verbatim: item.title)
                        .font(.title)
                        .bold()
                        .layoutPriority(1)

                    HStack(alignment: .firstTextBaseline) {
                        Text("Priority: ") +
                            Text(LocalizedStringKey(item.priority.rawValue.capitalized)).bold()
                        Spacer()
                        if item.date != nil {
                            Text("\(item.date!, formatter: Formatters.mediumDate)")
                        } else {
                            Text("No Due Date")
                        }
                    }
                    .font(.subheadline)
                }
                .foregroundColor(.white)
                .padding()
                .padding(.top)
                .background(headerBackground)
                .layoutPriority(1)
                
                if item.notes != nil {
                    Text(verbatim: self.item.notes!)
                        .padding()
                }
                Spacer()
            }
        }
        .navigationBarTitle("", displayMode: .inline)
        .navigationBarItems(trailing: editorButton)
    }
}

struct TodoItemDetail_Previews: PreviewProvider {
    static var previews: some View {
        single
            .environmentObject(DataCenter())
    }
    
    static var single: some View {
        NavigationView {
            TodoItemDetail(item: defaultTodoItems[0])
        }
    }
    
    static var allColorSchemePreviews: some View {
        ForEach(ColorScheme.allCases, id: \.self) { scheme in
            NavigationView {
                TodoItemDetail(item: defaultTodoItems[0])
            }
            .colorScheme(scheme)
            .previewDisplayName(String(describing: scheme))
        }
    }
    
    static var multiDevicePreviews: some View {
        ForEach(["iPhone 11", "iPhone SE"], id: \.self) { name in
            TodoItemDetail(item: defaultTodoItems[0])
                .previewDevice(PreviewDevice(rawValue: name))
                .previewDisplayName(name)
        }
    }
    
    static var allItemPreviews: some View {
        ForEach(defaultTodoItems) { item in
            TodoItemDetail(item: item)
        }
    }
    
    static var allLocalizations: some View {
        let localePairs: [(Locale, LayoutDirection)] = [
            (Locale(identifier: "en-US"), .leftToRight),
            (Locale(identifier: "pl"), .leftToRight),
            (Locale(identifier: "ar"), .rightToLeft),
        ]
        
        return ForEach(localePairs, id: \.self.0) { value in
            TodoItemDetail(item: defaultTodoItems[0])
                .environment(\.locale, value.0)
                .environment(\.layoutDirection, value.1)
                .previewDisplayName(value.0.languageCode ?? value.0.identifier)
        }
    }
    
    static var allSizeCategories: some View {
        ForEach(ContentSizeCategory.allCases, id: \.self) { category in
            TodoItemDetail(item: defaultTodoItems[0])
                .environment(\.sizeCategory, category)
                .previewDisplayName(String(describing: category))
        }
    }
}
